﻿using System;
using System.Reflection;
using System.Web;
using System.Web.Mvc;
using System.Web.Routing;
using Castle.MicroKernel.Registration;
using Castle.Windsor;
using easyhr.web.cache;
using easyhr.web.cache.stubs;
using easyhr.web.viewdata;
using it.messages;
using NServiceBus;
using utility.logging;

namespace easyhr.web.ui
{
    public class Global : HttpApplication
    {
        static public IBus Bus { get; private set; }

        protected void Application_Start()
        {
            AreaRegistration.RegisterAllAreas();
            RouteTable.Routes.IgnoreRoute("{resource}.axd/{*pathInfo}");
            RouteTable.Routes.MapRoute( "Default", "{controller}/{action}/{id}", new {controller = "home", action = "index", id = UrlParameter.Optional} );
            "starting web application".log();

            Bootstrap();
        }

        static void Bootstrap()
        {
            var container = new WindsorContainer();
            ControllerBuilder.Current.SetControllerFactory(new WindsorControllerFactory(container));

            container.Register(
                AllTypes
                    .FromAssembly(Assembly.GetExecutingAssembly())
                    .Where(t => t.Namespace.StartsWith("easyhr.web.ui.Controllers")).Configure(x => x.LifeStyle.Transient),
                    Component.For<IUsernameCache>().ImplementedBy<StubUsernameCache>().LifeStyle.Singleton,
                    Component.For<ICache<EmployeeViewData>>().ImplementedBy<WebCache<EmployeeViewData>>().LifeStyle.Singleton
                );

            Bus = Configure.WithWeb()
                .Log4Net()
                .CastleWindsorBuilder(container)
                .XmlSerializer()
                .RijndaelEncryptionService()
                .MsmqTransport().IsTransactional(true).PurgeOnStartup(true)
                .MsmqSubscriptionStorage()
                .UnicastBus().ImpersonateSender(true).LoadMessageHandlers()
                .CreateBus()
                .Start();
            Bus.Send<GetAllUserNamesQuery>(x => { });
        }

        class WindsorControllerFactory : DefaultControllerFactory
        {
            IWindsorContainer container;

            public WindsorControllerFactory(IWindsorContainer container)
            {
                this.container = container;
            }

            protected override IController GetControllerInstance(RequestContext context, Type controllerType)
            {
                if (controllerType == null)
                    throw new HttpException(404, string.Format("The controller for path '{0}' could not be found or it does not implement IController.", context.HttpContext.Request.Path));

                return (IController) container.Resolve(controllerType);
            }

            public override void ReleaseController(IController controller)
            {
                var disposable = controller as IDisposable;

                if (disposable != null)
                    disposable.Dispose();

                container.Release(controller);
            }
        }
    }
}