using CookComputing.XmlRpc;

namespace MetaWeblogApi {
	/// <summary> 
	/// This class can be used to programmatically interact with a Weblog on 
	/// MSN Spaces using the MetaWeblog API. 
	/// </summary> 
	[XmlRpcUrl( "https://storage.msn.com/storageservice/MetaWeblog.rpc" )]
	public class MsnSpacesMetaWeblog : XmlRpcClientProtocol {
		/// <summary> 
		/// Returns the most recent draft and non-draft blog posts sorted in descending order by publish date. 
		/// </summary> 
		/// <param name="blogid"> This should be the string MyBlog, which indicates that the post is being created in the users blog. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <param name="password"> The users secret word. </param> 
		/// <param name="numberOfPosts"> The number of posts to return. The maximum value is 20. </param> 
		/// <returns></returns> 
		[XmlRpcMethod( "metaWeblog.getRecentPosts" )]
		public Post[] getRecentPosts(
			string blogid,
			string username,
			string password,
			int numberOfPosts ) {
			return ( Post[] )Invoke( "getRecentPosts", new object[] {blogid, username, password, numberOfPosts} );
		}

		/// <summary> 
		/// Posts a new entry to a blog. 
		/// </summary> 
		/// <param name="blogid"> This should be the string MyBlog, which indicates that the post is being created in the users blog. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <param name="password"> The users secret word. </param> 
		/// <param name="content"> A struct representing the content to update. </param> 
		/// <param name="publish"> If false, this is a draft post. </param> 
		/// <returns> The postid of the newly-created post. </returns> 
		[XmlRpcMethod( "metaWeblog.newPost" )]
		public string newPost(
			string blogid,
			string username,
			string password,
			Post content,
			bool publish ) {
			return ( string )Invoke( "newPost", new object[] {blogid, username, password, content, publish} );
		}

		/// <summary> 
		/// Edits an existing entry on a blog. 
		/// </summary> 
		/// <param name="postid"> The ID of the post to update. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <param name="password"> The users secret word. </param> 
		/// <param name="content"> A struct representing the content to update. </param> 
		/// <param name="publish"> If false, this is a draft post. </param> 
		/// <returns> Always returns true. </returns> 
		[XmlRpcMethod( "metaWeblog.editPost" )]
		public bool editPost(
			string postid,
			string username,
			string password,
			Post content,
			bool publish ) {
			return ( bool )Invoke( "editPost", new object[] {postid, username, password, content, publish} );
		}

		/// <summary> 
		/// Deletes a post from the blog. 
		/// </summary> 
		/// <param name="appKey"> This value is ignored. </param> 
		/// <param name="postid"> The ID of the post to update. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <param name="password"> The users secret word. </param> 
		/// <param name="publish"> This value is ignored. </param> 
		/// <returns> Always returns true. </returns> 
		[XmlRpcMethod( "blogger.deletePost" )]
		public bool deletePost(
			string appKey,
			string postid,
			string username,
			string password,
			bool publish ) {
			return ( bool )Invoke( "deletePost", new object[] {appKey, postid, username, password, publish} );
		}

		/// <summary> 
		/// Returns information about the users space. An empty array is returned if the user does not have a space. 
		/// </summary> 
		/// <param name="appKey"> This value is ignored. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <returns> An array of structs that represents each of the users blogs. The array will contain a maximum of one struct, since a user can only have a single space with a single blog. </returns> 
		/// <param name="password"></param>
		[XmlRpcMethod( "blogger.getUsersBlogs" )]
		public UserBlog[] getUsersBlogs(
			string appKey,
			string username,
			string password ) {
			return ( UserBlog[] )Invoke( "getUsersBlogs", new object[] {appKey, username, password} );
		}

		/// <summary> 
		/// Returns basic user info (name, e-mail, userid, and so on). 
		/// </summary> 
		/// <param name="appKey"> This value is ignored. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <returns> A struct containing profile information about the user. 
		/// Each struct will contain the following fields: nickname, userid, url, e-mail, 
		/// lastname, and firstname. </returns> 
		/// <param name="password"></param>
		[XmlRpcMethod( "blogger.getUserInfo" )]
		public UserInfo getUserInfo(
			string appKey,
			string username,
			string password ) {
			return ( UserInfo )Invoke( "getUserInfo", new object[] {appKey, username, password} );
		}

		/// <summary> 
		/// Returns a specific entry from a blog. 
		/// </summary> 
		/// <param name="postid"> The ID of the post to update. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <param name="password"> The users secret word. </param> 
		/// <returns> Always returns true. </returns> 
		[XmlRpcMethod( "metaWeblog.getPost" )]
		public Post getPost(
			string postid,
			string username,
			string password ) {
			return ( Post )Invoke( "getPost", new object[] {postid, username, password} );
		}

		/// <summary> 
		/// Returns the list of categories that have been used in the blog. 
		/// </summary> 
		/// <param name="blogid"> This should be the string MyBlog, which indicates that the post is being created in the users blog. </param> 
		/// <param name="username"> The name of the users space. </param> 
		/// <param name="password"> The users secret word. </param> 
		/// <returns> An array of structs that contains one struct for each category. Each category struct will contain a description field that contains the name of the category. </returns> 
		[XmlRpcMethod( "metaWeblog.getCategories" )]
		public Category[] getCategories(
			string blogid,
			string username,
			string password ) {
			return ( Category[] )Invoke( "getCategories", new object[] {blogid, username, password} );
		}
	}
}