/*
 * Created by: Mo Khan
 * Created: Tuesday, May 08, 2007
 */
using System;
using System.Collections.Generic;
using System.IO;
using DesktopSearch.Domain;
using DesktopSearch.Task;
using DesktopSearch.Task.Google.Interop;
using NUnit.Framework;

namespace DesktopSearch.Test.Task.Interop
{
	[Explicit( "The Google Desktop SDK is not giving the expected behaviour." )]
	[TestFixture]
	public class GoogleSearchDocumentServiceTest
	{
		private ISearchDocumentService _task;

		[SetUp]
		public void SetUp( )
		{
			_task = new GoogleSearchDocumentService( );
		}

		[Test]
		public void Search_ShouldReturnCollectionOfSearchResults( )
		{
			const String searchFor = "1";
			ICollection< IDocument > documents = _task.Search( searchFor );
			Assert.IsNotNull( documents );
			Assert.IsTrue( documents.Count > 0 );
		}

		[Test]
		public void Search_ShouldReturnOnlyDocumentsInSpecifedDirectory( )
		{
			String directory = Environment.GetFolderPath( Environment.SpecialFolder.ProgramFiles );
			directory = directory.Replace( "/", @"\" );

			ICollection< IDocument > documents = _task.Search( directory, "Microsoft" );
			foreach( IDocument document in documents ) {
				document.FilePath = document.FilePath.Replace( "/", @"\" );
				StringAssert.Contains( directory, document.FilePath,
				                       "This document was not located in the directory we were searching from" );
			}
		}

		[Test]
		public void Search_ShouldReturnCorrectCreatedOnDateForDocument( )
		{
			ICollection< IDocument > documents = _task.Search( "microsoft" );
			foreach( IDocument document in documents ) {
				FileInfo file = new FileInfo( document.FilePath );
				StringAssert.IsMatch( file.CreationTimeUtc.ToString( ), document.CreatedDate.ToString( ) );
			}
		}

		[Test]
		public void Search_ShouldReturnCorrectModifiedDateForDocument( )
		{
			ICollection< IDocument > documents = _task.Search( "microsoft" );
			foreach( IDocument document in documents ) {
				FileInfo file = new FileInfo( document.FilePath );
				StringAssert.IsMatch( file.LastWriteTimeUtc.ToString( ), document.ModifiedDate.ToString( ) );
			}
		}

		[Test]
		public void SearchFor_ShouldReturnDocumentsWithMatchingFileName( )
		{
			const String FileName = "readme.txt";
			ICollection< IDocument > documents = _task.SearchFor( FileName );

			Assert.IsNotNull( documents );
			Assert.Greater( documents.Count, 0, "There should be at least one file." );
			foreach( IDocument document in documents ) {
				Assert.IsTrue( String.Compare( FileName, document.FileName, true ) == 0 );
			}
		}

		[Test]
		public void SearchFor_ShouldSearchForFileNameFromDirectory( )
		{
			const String fileName = "readme.txt";
			String directory = Environment.GetFolderPath( Environment.SpecialFolder.ProgramFiles );
			directory = directory.Replace( "/", @"\" );

			ICollection< IDocument > documents = _task.SearchFor( fileName, directory );

			Assert.IsNotNull( documents );
			Assert.Greater( documents.Count, 0, "There should be at least one file." );
			foreach( IDocument document in documents ) {
				document.FilePath = document.FilePath.Replace( "/", @"\" );
				StringAssert.Contains( directory, document.FilePath,
				                       "This document was not located in the directory we were searching from" );
				Assert.IsTrue( String.Compare( fileName, document.FileName, true ) == 0 );
			}
		}
	}
}