/*
 * Created by: Mo Khan
 * Created: Wednesday, May 09, 2007
 */
using System;
using System.Collections.Generic;
using DesktopSearch.Domain;
using DesktopSearch.Task.Google;
using NUnit.Framework;
using Rhino.Mocks;

namespace DesktopSearch.Test.Task
{
	[TestFixture]
	public class GoogleResultsParserTest
	{
		private MockRepository _mock;
		private IResultsParser _parser;

		[SetUp]
		public void SetUp( )
		{
			_mock = new MockRepository( );
			_parser = new ResultsParser( );
		}

		[TearDown]
		public void TearDown( )
		{
			_mock.VerifyAll( );
		}

		[Test]
		public void ParseResultsXml_ShouldReturnFiveResults( )
		{
			const String xmlResults =
				"<?xml version = \"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n\n<results count=\"10\">\n<result>\n<category>web</category>\n<doc_id>158349</doc_id>\n<event_id>158342</event_id>\n<title>MIT OpenCourseWare | Technical Requirements</title>\n<url>http://ocw.mit.edu/OcwWeb/Global/OCWHelp/technical-faq.htm#mp3</url>\n<flags>3</flags>\n<time>128226764941090000</time>\n<snippet>work with .dbf files, although they can also be opened with &lt;b&gt;Microsoft&lt;/b&gt; Excel software. dcd The DCD files are single precision binary FORTRAN files</snippet>\n<thumbnail>/thumbnail?id=6%5FVn1CDqvspcIAAAAA&amp;s=HPWgs9ycIO5ry66xdcIcDH7iy6A</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158342%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DIm6gatt3UOresmIBBPfcE5l%2DIvE&amp;src=1&amp;schema=2&amp;s=RHv2yREUSdTuiVc9xtO_Wz2jSg0</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158320</doc_id>\n<event_id>158315</event_id>\n<title>VisualSVN - Subversion plugin for Visual Studio</title>\n<url>http://www.visualsvn.com/</url>\n<flags>3</flags>\n<time>128226752703870000</time>\n<snippet>at your fingertips VisualSVN is a plugin for &lt;b&gt;Microsoft&lt;/b&gt; Visual Studio for seamless integration with Subversion .Be up</snippet>\n<thumbnail>/thumbnail?id=6%5FXdmXZhcrAd0AAAAA&amp;s=Yf2ggMKdDDLeUjaef59QBL5fpCI</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158315%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3Dtf6EU7cqkTQOneWdUveQwZiOoiw&amp;src=1&amp;schema=2&amp;s=Jx2TVkNAHEWpvJCG-J3b_0tgwvI</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158318</doc_id>\n<event_id>158313</event_id>\n<title>Creating Search Pages with Index Server and .NET - The Code Project - ASP..</title>\n<url>http://www.codeproject.com/aspnet/search.asp</url>\n<flags>3</flags>\n<time>128226752541400000</time>\n<snippet>you don't know -with the power and flexibility of &lt;b&gt;Microsoft&lt;/b&gt; Index Server and ASP.NET to create easy -yet powerful -custom</snippet>\n<thumbnail>/thumbnail?id=6%5FNpntBK6F8QQAAAAA&amp;s=F03v696jIKQndC7X2orBlinCJeQ</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158313%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DAmYqDVlBBZhj2e0a1d%2DwkHYAN3g&amp;src=1&amp;schema=2&amp;s=90kZ2eWODLtwie10kpDIpalXS_I</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98335</doc_id>\n<event_id>98322</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/VisualBasic/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264327010000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98322%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DrBioQEu8%2DLBLCm4tlUOba0xlPqE&amp;src=1&amp;schema=8&amp;s=XlzylfuapDyqYhLuYAPGHfr7p-E</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98232</doc_id>\n<event_id>98219</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/JSharp/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264302120000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98219%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DUXVeN0UW4An0R6xArtjvmeVUSuA&amp;src=1&amp;schema=8&amp;s=kwEOQ2zgtv5DGqmfupqmLGnXJ1Y</cache_url>\n</result>\n\n</results>\n\n";
			ICollection< GoogleSearchResult > results = _parser.ParseResultsXml( xmlResults );
			Assert.IsTrue( results.Count == 5, "Only 5 results should have been returned." );
		}

		[Test]
		public void Parse_ShouldReturnFiveResultsInAnEnumerator( )
		{
			const String xmlResults =
				"<?xml version = \"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n\n<results count=\"10\">\n<result>\n<category>web</category>\n<doc_id>158349</doc_id>\n<event_id>158342</event_id>\n<title>MIT OpenCourseWare | Technical Requirements</title>\n<url>http://ocw.mit.edu/OcwWeb/Global/OCWHelp/technical-faq.htm#mp3</url>\n<flags>3</flags>\n<time>128226764941090000</time>\n<snippet>work with .dbf files, although they can also be opened with &lt;b&gt;Microsoft&lt;/b&gt; Excel software. dcd The DCD files are single precision binary FORTRAN files</snippet>\n<thumbnail>/thumbnail?id=6%5FVn1CDqvspcIAAAAA&amp;s=HPWgs9ycIO5ry66xdcIcDH7iy6A</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158342%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DIm6gatt3UOresmIBBPfcE5l%2DIvE&amp;src=1&amp;schema=2&amp;s=RHv2yREUSdTuiVc9xtO_Wz2jSg0</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158320</doc_id>\n<event_id>158315</event_id>\n<title>VisualSVN - Subversion plugin for Visual Studio</title>\n<url>http://www.visualsvn.com/</url>\n<flags>3</flags>\n<time>128226752703870000</time>\n<snippet>at your fingertips VisualSVN is a plugin for &lt;b&gt;Microsoft&lt;/b&gt; Visual Studio for seamless integration with Subversion .Be up</snippet>\n<thumbnail>/thumbnail?id=6%5FXdmXZhcrAd0AAAAA&amp;s=Yf2ggMKdDDLeUjaef59QBL5fpCI</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158315%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3Dtf6EU7cqkTQOneWdUveQwZiOoiw&amp;src=1&amp;schema=2&amp;s=Jx2TVkNAHEWpvJCG-J3b_0tgwvI</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158318</doc_id>\n<event_id>158313</event_id>\n<title>Creating Search Pages with Index Server and .NET - The Code Project - ASP..</title>\n<url>http://www.codeproject.com/aspnet/search.asp</url>\n<flags>3</flags>\n<time>128226752541400000</time>\n<snippet>you don't know -with the power and flexibility of &lt;b&gt;Microsoft&lt;/b&gt; Index Server and ASP.NET to create easy -yet powerful -custom</snippet>\n<thumbnail>/thumbnail?id=6%5FNpntBK6F8QQAAAAA&amp;s=F03v696jIKQndC7X2orBlinCJeQ</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158313%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DAmYqDVlBBZhj2e0a1d%2DwkHYAN3g&amp;src=1&amp;schema=2&amp;s=90kZ2eWODLtwie10kpDIpalXS_I</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98335</doc_id>\n<event_id>98322</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/VisualBasic/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264327010000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98322%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DrBioQEu8%2DLBLCm4tlUOba0xlPqE&amp;src=1&amp;schema=8&amp;s=XlzylfuapDyqYhLuYAPGHfr7p-E</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98232</doc_id>\n<event_id>98219</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/JSharp/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264302120000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98219%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DUXVeN0UW4An0R6xArtjvmeVUSuA&amp;src=1&amp;schema=8&amp;s=kwEOQ2zgtv5DGqmfupqmLGnXJ1Y</cache_url>\n</result>\n\n</results>\n\n";
			IEnumerable< GoogleSearchResult > results = _parser.Parse( xmlResults );
			Int32 counter = 0;
			foreach( GoogleSearchResult result in results ) {
				++counter;
			}
			Assert.IsTrue( counter == 5, "Only 5 results should have been returned." );
		}

		[Test]
		public void ParseXmlResults_ShouldReturn10( )
		{
			const String xmlResults =
				"<?xml version = \"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n\n<results count=\"10\">\n<result>\n<category>file</category>\n<doc_id>162737</doc_id>\n<event_id>167729</event_id>\n<title>Vista Crack.zip</title>\n<url>C:\\Vista Crack.zip</url>\n<flags>2</flags>\n<time>128230517607270000</time>\n<snippet>cmd Vista Crack/vistaloader/&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt; Vista Crack/vistaloader/uninstall&lt;b&gt; .. &lt;/b&gt;zip Vista Crack/CD-KEY.&lt;b&gt;txt&lt;/b&gt; Vista Crack/vistaloader.rar Vista Crack</snippet>\n<icon>/file.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167729%26schema%5Fid%3D8%26q%3Dreadme%252Etxt%26s%3DSuhi05bK9G1GZJqqOtSj4d3%5FSjw&amp;src=1&amp;schema=8&amp;s=7yTIxNkzVZo3IdIRXnHRlyFME6M</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162821</doc_id>\n<event_id>167821</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Vista Crack\\vistaloader\\readme.txt</url>\n<flags>2</flags>\n<time>128230517522760000</time>\n<snippet>ġ-Vista Loader V2.0.0  Vista Loader Vista Loaderǵһͨͻ Vista Ƶйԭԭǣϵͳ֮ǰƵ ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167821%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3D%5F354CN%2DCs1Gey6TZOztAv2KoN%5FA&amp;src=1&amp;schema=4&amp;s=i_gwWKgtRGz3MsMtPZWxtnYjSIE</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>163071</doc_id>\n<event_id>168093</event_id>\n<title>Release Notes.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\Release Notes.txt</url>\n<flags>2</flags>\n<time>128230379574280000</time>\n<snippet>upgrade are in the &lt;b&gt;readme&lt;/b&gt;.rtf included with the package. This is a report from SF.NET on closed items up to this release. As with all projects there are always open bugs</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D168093%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3D8cryaGb3rSKss6NzfBrl0NA52Vo&amp;src=1&amp;schema=4&amp;s=_ZEqTNrWCG8so75L7mV-zYo1G4E</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162838</doc_id>\n<event_id>167840</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\Portal\\readme.txt</url>\n<flags>2</flags>\n<time>128230379560660000</time>\n<snippet>Portal theme for dasBlog by Johnny Hughes http:johnnynine.com/blog CHANGING THE STYLE SHEET -There are several style sheets included with this theme. Each style sheet ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167840%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DALXCYv3YBC5R%2D5iqUKcWaMsTNFw&amp;src=1&amp;schema=4&amp;s=vGlw687oSE6O4pXe4b2JfAb41rU</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162893</doc_id>\n<event_id>167895</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\essence\\readme.txt</url>\n<flags>2</flags>\n<time>128230379549830000</time>\n<snippet>Essence -A theme for dasBlog (http:www.dasblog.net) Created by Jelle Druyts (http:jelle.druyts.net) in 2005. Modify and redistribute as you please. With thanks to ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167895%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DXvwssaAAB8pc78X0T27yPEqCdcs&amp;src=1&amp;schema=4&amp;s=M8FzJYOam0JDHZ9Z5UrWP03AMyg</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162472</doc_id>\n<event_id>167450</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\elegante\\readme.txt</url>\n<flags>2</flags>\n<time>128230379549220000</time>\n<snippet>This theme, by default, uses a header graphic instead of the blog's 'Title' text from the site.config file. Since a blog's title doesn't change often (if ever) I ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167450%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DUsQfKE12yEvdUK4azxueDdUI69I&amp;src=1&amp;schema=4&amp;s=IKSidVnixHL0guh_KiUJzmVG1J8</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162874</doc_id>\n<event_id>167876</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\blogxp\\readme.txt</url>\n<flags>2</flags>\n<time>128230379534520000</time>\n<snippet>BlogXP -A theme for dasBlog (http:www.dasblog.net) Created by Jelle Druyts (http:jelle.druyts.net) in 2005. Images reused and adapted from http:www</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167876%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DGZAQsdZ0tArY73Lqaf%2DblQNqoDE&amp;src=1&amp;schema=4&amp;s=AqN-vLMn7luN6IzGWX2gVfxrSnM</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>163077</doc_id>\n<event_id>168104</event_id>\n<title>&lt;b&gt;ReadMe&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\DatePicker\\ReadMe.txt</url>\n<flags>2</flags>\n<time>128230379508680000</time>\n<snippet>~ BDP Lite v1.2 ~Welcome DasBlog users! BDP Lite is a professional cross-browser ASP.NET date picker calendar web control. This quick loading DHTML calendar enables ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D168104%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3D5mUFF%2DgPtr%5FYj%2DPULS4hmr%5Fkn3o&amp;src=1&amp;schema=4&amp;s=JY_wRoPUDY1jMWkeGqQQ4ZLgaL8</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>161568</doc_id>\n<event_id>167060</event_id>\n<title>dasBlog-1.9.7067.0-Web-Files.zip</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files.zip</url>\n<flags>2</flags>\n<time>128230379370370000</time>\n<snippet>BDPLite.js dasblogce/DatePicker/button.gif dasblogce/DatePicker/&lt;b&gt;ReadMe&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt; dasblogce/default.aspx dasblogce/EditBlogRoll.aspx dasblogce/EditBlogRollBox</snippet>\n<icon>/file.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167060%26schema%5Fid%3D8%26q%3Dreadme%252Etxt%26s%3D9Kbz84Webmtp2f7SVhw05W1339I&amp;src=1&amp;schema=8&amp;s=hUDFLjvOEheueDmFtoh0LEdip8Q</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162267</doc_id>\n<event_id>167234</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>archive://C:/Misc/dasBlog-1.9.7067.0-Web-Files.zip/dasblogce/themes/essence/readme.txt</url>\n<flags>2</flags>\n<time>128227312200000000</time>\n<snippet>Essence -A theme for dasBlog (http:www.dasblog.net) Created by Jelle Druyts (http:jelle.druyts.net) in 2005. Modify and redistribute as you please. With thanks to ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167234%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3Daqoytlbu9OkTcJ6JVmw9PhucASA&amp;src=1&amp;schema=4&amp;s=hGD_5dRht9vSXeuoe20sfKXV7HQ</cache_url>\n</result>\n\n</results>\n\n";
			ICollection< GoogleSearchResult > results = _parser.ParseResultsXml( xmlResults );
			Assert.IsTrue( results.Count == 10, "10 results should have been returned from the parser." );
		}

		[Test]
		public void Parse_ShouldReturn10ResultsInAnEnumerator( )
		{
			const String xmlResults =
				"<?xml version = \"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n\n<results count=\"10\">\n<result>\n<category>file</category>\n<doc_id>162737</doc_id>\n<event_id>167729</event_id>\n<title>Vista Crack.zip</title>\n<url>C:\\Vista Crack.zip</url>\n<flags>2</flags>\n<time>128230517607270000</time>\n<snippet>cmd Vista Crack/vistaloader/&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt; Vista Crack/vistaloader/uninstall&lt;b&gt; .. &lt;/b&gt;zip Vista Crack/CD-KEY.&lt;b&gt;txt&lt;/b&gt; Vista Crack/vistaloader.rar Vista Crack</snippet>\n<icon>/file.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167729%26schema%5Fid%3D8%26q%3Dreadme%252Etxt%26s%3DSuhi05bK9G1GZJqqOtSj4d3%5FSjw&amp;src=1&amp;schema=8&amp;s=7yTIxNkzVZo3IdIRXnHRlyFME6M</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162821</doc_id>\n<event_id>167821</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Vista Crack\\vistaloader\\readme.txt</url>\n<flags>2</flags>\n<time>128230517522760000</time>\n<snippet>ġ-Vista Loader V2.0.0  Vista Loader Vista Loaderǵһͨͻ Vista Ƶйԭԭǣϵͳ֮ǰƵ ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167821%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3D%5F354CN%2DCs1Gey6TZOztAv2KoN%5FA&amp;src=1&amp;schema=4&amp;s=i_gwWKgtRGz3MsMtPZWxtnYjSIE</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>163071</doc_id>\n<event_id>168093</event_id>\n<title>Release Notes.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\Release Notes.txt</url>\n<flags>2</flags>\n<time>128230379574280000</time>\n<snippet>upgrade are in the &lt;b&gt;readme&lt;/b&gt;.rtf included with the package. This is a report from SF.NET on closed items up to this release. As with all projects there are always open bugs</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D168093%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3D8cryaGb3rSKss6NzfBrl0NA52Vo&amp;src=1&amp;schema=4&amp;s=_ZEqTNrWCG8so75L7mV-zYo1G4E</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162838</doc_id>\n<event_id>167840</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\Portal\\readme.txt</url>\n<flags>2</flags>\n<time>128230379560660000</time>\n<snippet>Portal theme for dasBlog by Johnny Hughes http:johnnynine.com/blog CHANGING THE STYLE SHEET -There are several style sheets included with this theme. Each style sheet ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167840%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DALXCYv3YBC5R%2D5iqUKcWaMsTNFw&amp;src=1&amp;schema=4&amp;s=vGlw687oSE6O4pXe4b2JfAb41rU</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162893</doc_id>\n<event_id>167895</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\essence\\readme.txt</url>\n<flags>2</flags>\n<time>128230379549830000</time>\n<snippet>Essence -A theme for dasBlog (http:www.dasblog.net) Created by Jelle Druyts (http:jelle.druyts.net) in 2005. Modify and redistribute as you please. With thanks to ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167895%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DXvwssaAAB8pc78X0T27yPEqCdcs&amp;src=1&amp;schema=4&amp;s=M8FzJYOam0JDHZ9Z5UrWP03AMyg</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162472</doc_id>\n<event_id>167450</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\elegante\\readme.txt</url>\n<flags>2</flags>\n<time>128230379549220000</time>\n<snippet>This theme, by default, uses a header graphic instead of the blog's 'Title' text from the site.config file. Since a blog's title doesn't change often (if ever) I ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167450%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DUsQfKE12yEvdUK4azxueDdUI69I&amp;src=1&amp;schema=4&amp;s=IKSidVnixHL0guh_KiUJzmVG1J8</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162874</doc_id>\n<event_id>167876</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\themes\\blogxp\\readme.txt</url>\n<flags>2</flags>\n<time>128230379534520000</time>\n<snippet>BlogXP -A theme for dasBlog (http:www.dasblog.net) Created by Jelle Druyts (http:jelle.druyts.net) in 2005. Images reused and adapted from http:www</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167876%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3DGZAQsdZ0tArY73Lqaf%2DblQNqoDE&amp;src=1&amp;schema=4&amp;s=AqN-vLMn7luN6IzGWX2gVfxrSnM</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>163077</doc_id>\n<event_id>168104</event_id>\n<title>&lt;b&gt;ReadMe&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files\\dasblogce\\DatePicker\\ReadMe.txt</url>\n<flags>2</flags>\n<time>128230379508680000</time>\n<snippet>~ BDP Lite v1.2 ~Welcome DasBlog users! BDP Lite is a professional cross-browser ASP.NET date picker calendar web control. This quick loading DHTML calendar enables ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D168104%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3D5mUFF%2DgPtr%5FYj%2DPULS4hmr%5Fkn3o&amp;src=1&amp;schema=4&amp;s=JY_wRoPUDY1jMWkeGqQQ4ZLgaL8</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>161568</doc_id>\n<event_id>167060</event_id>\n<title>dasBlog-1.9.7067.0-Web-Files.zip</title>\n<url>C:\\Misc\\dasBlog-1.9.7067.0-Web-Files.zip</url>\n<flags>2</flags>\n<time>128230379370370000</time>\n<snippet>BDPLite.js dasblogce/DatePicker/button.gif dasblogce/DatePicker/&lt;b&gt;ReadMe&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt; dasblogce/default.aspx dasblogce/EditBlogRoll.aspx dasblogce/EditBlogRollBox</snippet>\n<icon>/file.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167060%26schema%5Fid%3D8%26q%3Dreadme%252Etxt%26s%3D9Kbz84Webmtp2f7SVhw05W1339I&amp;src=1&amp;schema=8&amp;s=hUDFLjvOEheueDmFtoh0LEdip8Q</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>162267</doc_id>\n<event_id>167234</event_id>\n<title>&lt;b&gt;readme&lt;/b&gt;.&lt;b&gt;txt&lt;/b&gt;</title>\n<url>archive://C:/Misc/dasBlog-1.9.7067.0-Web-Files.zip/dasblogce/themes/essence/readme.txt</url>\n<flags>2</flags>\n<time>128227312200000000</time>\n<snippet>Essence -A theme for dasBlog (http:www.dasblog.net) Created by Jelle Druyts (http:jelle.druyts.net) in 2005. Modify and redistribute as you please. With thanks to ..</snippet>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D167234%26schema%5Fid%3D4%26q%3Dreadme%252Etxt%26s%3Daqoytlbu9OkTcJ6JVmw9PhucASA&amp;src=1&amp;schema=4&amp;s=hGD_5dRht9vSXeuoe20sfKXV7HQ</cache_url>\n</result>\n\n</results>\n\n";
			IEnumerable< GoogleSearchResult > results = _parser.Parse( xmlResults );
			Int32 counter = 0;
			foreach( GoogleSearchResult result in results ) {
				++counter;
			}
			Assert.IsTrue( counter == 10, "10 results should have been returned from the parser." );
		}

		[Test]
		public void ParseResult_ShouldReturnNullIfEmptyStringPassedInAsInputParameter( )
		{
			Assert.IsNull( _parser.ParseSingle( String.Empty ), "A Null value should have been returned." );
		}

		[Test]
		public void ParseResult_ShouldReturnNullIfNullPassedInAsInputParameter( )
		{
			Assert.IsNull( _parser.ParseSingle( null ), "A null value should have been returned" );
		}
	}
}