/*
 * Created by: Mo Khan
 * Created: Thursday, May 10, 2007
 */
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using DesktopSearch.Domain;
using DesktopSearch.Task.Google;
using NUnit.Framework;
using Rhino.Mocks;

namespace DesktopSearch.Test.Task
{
	[TestFixture]
	public class GoogleResultsConverterTest
	{
		private MockRepository _mock;
		private ResultsConverter _converter;
		private GoogleSearchResult _result;
		private const String Snippet = "foo";

		[SetUp]
		public void SetUp( )
		{
			_mock = new MockRepository( );
			_result = _mock.CreateMock< GoogleSearchResult >( );
			_converter = new ResultsConverter( );
		}

		[TearDown]
		public void TearDown( )
		{
			_mock.VerifyAll( );
		}

		[Test]
		public void Convert_ShouldReturnADocumentWithMathingValuesParsedFromSearchResult( )
		{
			String tempFilePath = Path.GetTempFileName( );
			_result.Url = new Uri( tempFilePath );
			_result.Snippet = "Just a taste";
			_mock.ReplayAll( );

			IDocument document = _converter.Convert< GoogleSearchResult >( _result );
			StringAssert.IsMatch( "Just a taste", document.Summary );
			Assert.IsTrue( document.FilePath == tempFilePath );
		}

		[Test]
		public void Convert_ShouldReturnEnumatorOfDocuments( )
		{
			IEnumerable< IGoogleSearchResult > results = GetMockResults( );
			_mock.ReplayAll( );

			IEnumerable< IDocument > documents = _converter.Convert( results );
			foreach( IDocument document in documents ) {
				StringAssert.IsMatch( Snippet, document.Summary );
			}
		}

		[Test]
		public void ConvertFilesNamed_ShouldOnlyReturnNullBecauseNoFilesShouldHaveTheNameWeAreSearchingFor( )
		{
			IEnumerable< IGoogleSearchResult > results = GetMockResults( );
			_mock.ReplayAll( );

			String fileName = Path.GetTempFileName( );
			IEnumerable< IDocument > documents = _converter.ConvertFilesNamed( results, fileName );
			foreach( IDocument document in documents ) {
				Assert.IsNull( document );
			}
		}

		[Test]
		public void ConvertFilesNamed_ShouldOnlyReturnOneDocumentsWithSameFileName( )
		{
			String filePath = Path.GetTempFileName( );
			ICollection< IGoogleSearchResult > results = GetMockResults( ) as ICollection< IGoogleSearchResult >;
			GoogleSearchResult result = new GoogleSearchResult( );
			result.Url = new Uri( filePath );
			_mock.ReplayAll( );

			IEnumerable< IDocument > documents = _converter.ConvertFilesNamed( results, filePath );
			foreach( IDocument document in documents ) {
				Assert.IsTrue( filePath == document.FilePath );
			}
		}

		[Test]
		public void Convert_ShouldReturn5DocumentsInCollection( )
		{
			_mock.ReplayAll( );
			const String xmlResults =
				"<?xml version = \"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n\n<results count=\"5\">\n<result>\n<category>web</category>\n<doc_id>158349</doc_id>\n<event_id>158342</event_id>\n<title>MIT OpenCourseWare | Technical Requirements</title>\n<url>http://ocw.mit.edu/OcwWeb/Global/OCWHelp/technical-faq.htm#mp3</url>\n<flags>3</flags>\n<time>128226764941090000</time>\n<snippet>work with .dbf files, although they can also be opened with &lt;b&gt;Microsoft&lt;/b&gt; Excel software. dcd The DCD files are single precision binary FORTRAN files</snippet>\n<thumbnail>/thumbnail?id=6%5FVn1CDqvspcIAAAAA&amp;s=HPWgs9ycIO5ry66xdcIcDH7iy6A</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158342%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DIm6gatt3UOresmIBBPfcE5l%2DIvE&amp;src=1&amp;schema=2&amp;s=RHv2yREUSdTuiVc9xtO_Wz2jSg0</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158320</doc_id>\n<event_id>158315</event_id>\n<title>VisualSVN - Subversion plugin for Visual Studio</title>\n<url>http://www.visualsvn.com/</url>\n<flags>3</flags>\n<time>128226752703870000</time>\n<snippet>at your fingertips VisualSVN is a plugin for &lt;b&gt;Microsoft&lt;/b&gt; Visual Studio for seamless integration with Subversion .Be up</snippet>\n<thumbnail>/thumbnail?id=6%5FXdmXZhcrAd0AAAAA&amp;s=Yf2ggMKdDDLeUjaef59QBL5fpCI</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158315%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3Dtf6EU7cqkTQOneWdUveQwZiOoiw&amp;src=1&amp;schema=2&amp;s=Jx2TVkNAHEWpvJCG-J3b_0tgwvI</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158318</doc_id>\n<event_id>158313</event_id>\n<title>Creating Search Pages with Index Server and .NET - The Code Project - ASP..</title>\n<url>http://www.codeproject.com/aspnet/search.asp</url>\n<flags>3</flags>\n<time>128226752541400000</time>\n<snippet>you don't know -with the power and flexibility of &lt;b&gt;Microsoft&lt;/b&gt; Index Server and ASP.NET to create easy -yet powerful -custom</snippet>\n<thumbnail>/thumbnail?id=6%5FNpntBK6F8QQAAAAA&amp;s=F03v696jIKQndC7X2orBlinCJeQ</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158313%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DAmYqDVlBBZhj2e0a1d%2DwkHYAN3g&amp;src=1&amp;schema=2&amp;s=90kZ2eWODLtwie10kpDIpalXS_I</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98335</doc_id>\n<event_id>98322</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/VisualBasic/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264327010000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98322%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DrBioQEu8%2DLBLCm4tlUOba0xlPqE&amp;src=1&amp;schema=8&amp;s=XlzylfuapDyqYhLuYAPGHfr7p-E</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98232</doc_id>\n<event_id>98219</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/JSharp/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264302120000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98219%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DUXVeN0UW4An0R6xArtjvmeVUSuA&amp;src=1&amp;schema=8&amp;s=kwEOQ2zgtv5DGqmfupqmLGnXJ1Y</cache_url>\n</result>\n\n</results>\n\n";
			IResultsParser parser = new ResultsParser( );
			ICollection< GoogleSearchResult > results = parser.ParseResultsXml( xmlResults );

			ICollection< IDocument > documents = _converter.ConvertToCollection( results );
			Assert.IsTrue( documents.Count == 5 );
		}

		[Test]
		public void Convert_ShouldReturnOnlyOneResultWithMatchingFileName( )
		{
			_mock.ReplayAll( );
			const String xmlResults =
				"<?xml version = \"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n\n<results count=\"5\">\n<result>\n<category>web</category>\n<doc_id>158349</doc_id>\n<event_id>158342</event_id>\n<title>MIT OpenCourseWare | Technical Requirements</title>\n<url>http://ocw.mit.edu/OcwWeb/Global/OCWHelp/technical-faq.htm#mp3</url>\n<flags>3</flags>\n<time>128226764941090000</time>\n<snippet>work with .dbf files, although they can also be opened with &lt;b&gt;Microsoft&lt;/b&gt; Excel software. dcd The DCD files are single precision binary FORTRAN files</snippet>\n<thumbnail>/thumbnail?id=6%5FVn1CDqvspcIAAAAA&amp;s=HPWgs9ycIO5ry66xdcIcDH7iy6A</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158342%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DIm6gatt3UOresmIBBPfcE5l%2DIvE&amp;src=1&amp;schema=2&amp;s=RHv2yREUSdTuiVc9xtO_Wz2jSg0</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158320</doc_id>\n<event_id>158315</event_id>\n<title>VisualSVN - Subversion plugin for Visual Studio</title>\n<url>http://www.visualsvn.com/</url>\n<flags>3</flags>\n<time>128226752703870000</time>\n<snippet>at your fingertips VisualSVN is a plugin for &lt;b&gt;Microsoft&lt;/b&gt; Visual Studio for seamless integration with Subversion .Be up</snippet>\n<thumbnail>/thumbnail?id=6%5FXdmXZhcrAd0AAAAA&amp;s=Yf2ggMKdDDLeUjaef59QBL5fpCI</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158315%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3Dtf6EU7cqkTQOneWdUveQwZiOoiw&amp;src=1&amp;schema=2&amp;s=Jx2TVkNAHEWpvJCG-J3b_0tgwvI</cache_url>\n</result>\n\n<result>\n<category>web</category>\n<doc_id>158318</doc_id>\n<event_id>158313</event_id>\n<title>Creating Search Pages with Index Server and .NET - The Code Project - ASP..</title>\n<url>http://www.codeproject.com/aspnet/search.asp</url>\n<flags>3</flags>\n<time>128226752541400000</time>\n<snippet>you don't know -with the power and flexibility of &lt;b&gt;Microsoft&lt;/b&gt; Index Server and ASP.NET to create easy -yet powerful -custom</snippet>\n<thumbnail>/thumbnail?id=6%5FNpntBK6F8QQAAAAA&amp;s=F03v696jIKQndC7X2orBlinCJeQ</thumbnail>\n<icon>/html.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D158313%26schema%5Fid%3D2%26q%3DMicrosoft%26s%3DAmYqDVlBBZhj2e0a1d%2DwkHYAN3g&amp;src=1&amp;schema=2&amp;s=90kZ2eWODLtwie10kpDIpalXS_I</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98335</doc_id>\n<event_id>98322</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/VisualBasic/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264327010000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98322%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DrBioQEu8%2DLBLCm4tlUOba0xlPqE&amp;src=1&amp;schema=8&amp;s=XlzylfuapDyqYhLuYAPGHfr7p-E</cache_url>\n</result>\n\n<result>\n<category>file</category>\n<doc_id>98232</doc_id>\n<event_id>98219</event_id>\n<title>textfile.txt</title>\n<url>archive://C:/Program Files/Microsoft Visual Studio 8/Common7/IDE/ItemTemplates/Web/JSharp/1033/TextFile.zip/textfile.txt</url>\n<flags>0</flags>\n<time>128226264302120000</time>\n<icon>/txt.gif</icon>\n<cache_url>http://127.0.0.1:4664/redir?url=http%3A%2F%2F127%2E0%2E0%2E1%3A4664%2Fcache%3Fevent%5Fid%3D98219%26schema%5Fid%3D8%26q%3DMicrosoft%26s%3DUXVeN0UW4An0R6xArtjvmeVUSuA&amp;src=1&amp;schema=8&amp;s=kwEOQ2zgtv5DGqmfupqmLGnXJ1Y</cache_url>\n</result>\n\n</results>\n\n";
			IResultsParser parser = new ResultsParser( );
			ICollection< GoogleSearchResult > results = parser.ParseResultsXml( xmlResults );

			ICollection< IDocument > documents = _converter.ConvertToCollection( results, "technical-faq.htm#mp3" );
			foreach( IDocument document in documents ) {
				StringAssert.IsMatch( "technical-faq.htm#mp3", document.FileName );
			}
		}

		#region Private Methods

		private IEnumerable< IGoogleSearchResult > GetMockResults( )
		{
			Collection< IGoogleSearchResult > results = new Collection< IGoogleSearchResult >( );
			for( Int32 i__ = 0; i__ < 5; i__++ ) {
				GoogleSearchResult result = _mock.CreateMock< GoogleSearchResult >( );
				result.Snippet = Snippet;
				result.Url = new Uri( Path.GetTempFileName( ) );
				results.Add( result );
			}

			return results;
		}

		#endregion
	}
}