/*
 * Created by: Mo Khan
 * Created: Monday, May 07, 2007
 */

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using DesktopSearch.Domain;
using DesktopSearch.Presentation;
using DesktopSearch.Task;
using NUnit.Framework;
using Rhino.Mocks;
using Rhino.Mocks.Constraints;
using Rhino.Mocks.Interfaces;

namespace DesktopSearch.Test.Presentation
{
	[TestFixture]
	public class SearchPresenterTest
	{
		private MockRepository _mock;
		private ISearchView _view;
		private ISearchDocumentService _task;
		private IEventRaiser _searchClickRaiser;
		private IEventRaiser _loadRaiser;
		//private IAppSettings _settings;

		[SetUp]
		public void Setup( )
		{
			_mock = new MockRepository( );
			_task = _mock.CreateMock< ISearchDocumentService >( );
			_view = _mock.CreateMock< ISearchView >( );
			//_settings = _mock.CreateMock< IAppSettings >( );

			Expect.Call( _view.FileNameSearch ).PropertyBehavior( );
			_view.FileNameSearch = false;
			Expect.Call( _view.TotalResults ).PropertyBehavior( );
			Expect.Call( _view.Results ).PropertyBehavior( );
			Expect.Call( _view.FromDirectory ).PropertyBehavior( );
			_view.FromDirectory = false;

			// wire up the search button click
			_view.SearchButtonClick += null;
			LastCall.Constraints( Is.NotNull( ) );
			_searchClickRaiser = LastCall.GetEventRaiser( );

			// wire up Load Event
			_view.Load += null;
			LastCall.Constraints( Is.NotNull( ) );
			_loadRaiser = LastCall.GetEventRaiser( );
		}

		[TearDown]
		public void TearDown( )
		{
			_mock.VerifyAll( );
		}

		[Test]
		public void Ctor_ShouldWireUpSearchButtonClickEventHandler( )
		{
			_mock.ReplayAll( );
			new SearchPresenter( _view, _task );
		}

		[Test]
		public void Search_ShouldSearchForTextInSearchTextbox( )
		{
			const String SearchText = "Foo";
			Expect.Call( _view.SearchText ).Return( SearchText ).Repeat.AtLeastOnce( );
			Expect.Call( _task.Search( SearchText ) ).Return( null ).Repeat.Once( );

			_mock.ReplayAll( );

			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.Search( );
		}

		[Test]
		public void Search_ShouldSetNumberOfResultsTextToFive( )
		{
			const String SearchText = "Bar";
			ICollection< IDocument > documents = GetDocuments( 5 );

			Expect.Call( _view.SearchText ).Return( SearchText ).Repeat.AtLeastOnce( );
			Expect.Call( _task.Search( SearchText ) ).Return( documents ).Repeat.Once( );
			_view.TotalResults = 5;

			_mock.ReplayAll( );
			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.Search( );
		}

		[Test]
		public void Search_ShouldCheckIfFileNameSearchIsEnabled( )
		{
			String fileName = Path.GetTempFileName( );

			Expect.Call( _view.SearchText ).Return( fileName ).Repeat.AtLeastOnce( );
			Expect.Call( _task.SearchFor( fileName ) ).Return( null ).Repeat.AtLeastOnce( );
			_view.FileNameSearch = true;

			_mock.ReplayAll( );
			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.Search( );
		}

		[Test]
		public void DataBind_ShouldReturnTableWithFiveColumns( )
		{
			ICollection< IDocument > documents = GetDocuments( 5 );
			_view.DataBind( );

			_mock.ReplayAll( );

			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.DataBind( documents );
			Assert.IsNotNull( _view.Results, "The search results table should not be null" );
			Assert.IsNotNull( _view.Results.Columns[ SearchPresenterColumns.Title ], "Title column should be set" );
			Assert.IsNotNull( _view.Results.Columns[ SearchPresenterColumns.Path ], "Path column should be set" );
			Assert.IsNotNull( _view.Results.Columns[ SearchPresenterColumns.SampleText ], "SampleText column should be set" );
			Assert.IsNotNull( _view.Results.Columns[ SearchPresenterColumns.CreatedDate ], "Created date column should be set" );
			Assert.IsNotNull( _view.Results.Columns[ SearchPresenterColumns.ModifiedDate ], "Modified date column should be set" );
		}

		[Test]
		public void DataBind_ShouldReturnFiveRows( )
		{
			ICollection< IDocument > documents = GetDocuments( 5 );
			_view.DataBind( );
			_mock.ReplayAll( );

			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.DataBind( documents );
			Assert.IsTrue( _view.Results.Rows.Count == 5, "The results table does not contain 5 rows!" );
			Int32 i__ = 0;
			foreach( IDocument document in documents ) {
				Assert.AreEqual( document.FileName, _view.Results.Rows[ i__ ][ SearchPresenterColumns.Title ].ToString( ),
				                 "filename does not match for {0}", document.FileName );
				StringAssert.IsMatch( document.CreatedDate.ToString( ),
				                      _view.Results.Rows[ i__ ][ SearchPresenterColumns.CreatedDate ].ToString( ),
				                      "created date does not match for {0}", document.FileName );
				StringAssert.IsMatch( document.ModifiedDate.ToString( ),
				                      _view.Results.Rows[ i__ ][ SearchPresenterColumns.ModifiedDate ].ToString( ),
				                      "modified date does not match for {0}", document.FileName );
				StringAssert.Contains( document.FilePath.ToString( ),
				                       _view.Results.Rows[ i__ ][ SearchPresenterColumns.Path ].ToString( ),
				                       "file path does not contain partial path for {0}", document.FilePath );
				StringAssert.IsMatch( document.Summary.ToString( ),
				                      _view.Results.Rows[ i__ ][ SearchPresenterColumns.SampleText ].ToString( ),
				                      "summary does not match for {0}", document.FileName );
				++i__;
			}
		}

		[Test]
		public void DataBind_ShouldCallViewsDatabind( )
		{
			ICollection< IDocument > documents = GetDocuments( 5 );
			_view.DataBind( );
			LastCall.Repeat.Once( );
			_mock.ReplayAll( );

			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.DataBind( documents );
		}

		[Test]
		public void DataBind_ShouldReturnNull( )
		{
			_view.DataBind( );
			LastCall.Repeat.Once( );
			_mock.ReplayAll( );

			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.DataBind( null );
			Assert.IsNull( _view.Results );
		}

		[Test]
		public void Search_ShouldCheckIfDirectoryFilterEnabledForTextSearch( )
		{
			String directory = Path.GetTempPath( );
			_view.FromDirectory = true;
			//Expect.Call( _view.FromDirectory ).Repeat.AtLeastOnce( );

			Expect.Call( _view.SearchInDirectory ).Return( directory ).Repeat.AtLeastOnce( );
			Expect.Call( _view.SearchText ).Return( "Microsoft" ).Repeat.AtLeastOnce( );
			Expect.Call( _task.Search( "Microsoft", directory ) ).Return( null );
			_mock.ReplayAll( );

			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.Search( );
		}

		[Test]
		public void Search_ShouldCheckIfDirectoryFilterEnabledForFileNameSearch( )
		{
			String directory = Environment.GetFolderPath( Environment.SpecialFolder.ProgramFiles );
			String fileName = "readme.txt";

			_view.FromDirectory = true;
			_view.FileNameSearch = true;

			Expect.Call( _view.SearchInDirectory ).Return( directory ).Repeat.AtLeastOnce( );
			Expect.Call( _view.SearchText ).Return( fileName ).Repeat.AtLeastOnce( );
			Expect.Call( _task.SearchFor( fileName, directory ) ).Return( null );
			_mock.ReplayAll( );

			SearchPresenter presenter = new SearchPresenter( _view, _task );
			presenter.Search( );
		}

		[Test]
		public void SearchButtonClick_ShouldSearchAndBindData( )
		{
			_view.DataBind( );
			Expect.Call( _view.SearchText ).Return( "Microsoft" );
			Expect.Call( _task.Search( "Microsoft" ) ).Return( null );
			_mock.ReplayAll( );

			new SearchPresenter( _view, _task );
			_searchClickRaiser.Raise( null, null );
		}

		#region Private Methods

		private ICollection< IDocument > GetDocuments( Int32 count )
		{
			ICollection< IDocument > documents = new Collection< IDocument >( );
			for( Int32 i__ = 0; i__ < count; i__++ ) {
				IDocument document = _mock.CreateMock< IDocument >( );
				Expect.Call( document.CreatedDate ).Return( DateTime.Now.AddDays( -1 ) ).Repeat.Any( );
				Expect.Call( document.ModifiedDate ).Return( DateTime.Now.AddHours( -1 ) ).Repeat.Any( );
				Expect.Call( document.FilePath ).Return( Path.GetTempPath( ) + Path.PathSeparator + Path.GetTempFileName( ) ).Repeat
					.Any( );
				Expect.Call( document.Summary ).Return( "This is a mock summary" ).Repeat.Any( );
				Expect.Call( document.FileName ).Return( Path.GetTempFileName( ) ).Repeat.Any( );
				documents.Add( document );
			}

			return documents;
		}

		#endregion
	}
}