using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using DesktopSearch.Domain;
using DesktopSearch.Task.Google;

namespace DesktopSearch.Task
{
	public class SearchDocumentService : ISearchDocumentService
	{
		#region Constructors

		public SearchDocumentService( ) : this( new ResultsParser( ) ) {}

		public SearchDocumentService( IResultsParser parser )
		{
			_parser = parser;
		}

		#endregion

		#region Public Methods

		/// <summary>Searches for text in documents.</summary>
		/// <param name="forText">The text to search for in the indexed documents.</param>
		/// <returns>A collection of documents found containing the text to search for.</returns>
		public ICollection< IDocument > Search( String forText )
		{
			return Search( forText, null );
		}

		public ICollection< IDocument > Search( String forText, String fromDirectory )
		{
			return GetDocuments( forText, fromDirectory, SearchType.Text );
		}

		public ICollection< IDocument > SearchFor( String fileName )
		{
			return SearchFor( fileName, null );
		}

		public ICollection< IDocument > SearchFor( String fileName, String fromDirectory )
		{
			return GetDocuments( fileName, fromDirectory, SearchType.FileName );
		}

		#endregion

		#region Private Fields

		private IResultsParser _parser;

		private enum SearchType
		{
			FileName,
			Text
		}

		#endregion

		#region Private Methods

		private static String QueryGoogleDesktop( String searchQuery )
		{
			String responseXml;
			WebClient googleRequest = new WebClient( );
			using( StreamReader stream = new StreamReader( googleRequest.OpenRead( searchQuery ) ) ) {
				responseXml = stream.ReadToEnd( );
			}
			return responseXml;
		}

		private ICollection< IDocument > GetDocuments( String searchFor, String fromDirectory, SearchType filter )
		{
			QueryBuilder builder = new QueryBuilder( );
			if( !String.IsNullOrEmpty( fromDirectory ) ) {
				builder.AddFilter( fromDirectory );
			}

			String xmlResults = QueryGoogleDesktop( builder.Build( searchFor ) );
			ICollection< GoogleSearchResult > results = _parser.ParseResultsXml( xmlResults );
			if( null == results ) {
				return null;
			}

			ResultsConverter converter = new ResultsConverter( );
			if( SearchType.FileName == filter ) {
				return converter.ConvertToCollection( results, searchFor );
			}
			return converter.ConvertToCollection( results );
		}

		#endregion
	}
}