/*
 * Created by: Mo Khan
 * Created: Thursday, May 10, 2007
 */

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using DesktopSearch.Domain;

namespace DesktopSearch.Task.Google
{
	public class ResultsConverter
	{
		#region Public Methods

		public IDocument Convert( GoogleSearchResult result )
		{
			FileDates dates = RetrieveDateUtc( result.Url.LocalPath );
			if( File.Exists( result.Url.LocalPath ) ) {
				FileInfo info = new FileInfo( result.Url.LocalPath );
				return new Document( info.Name,
				                     result.Url.LocalPath,
				                     result.Snippet,
				                     dates.Created,
				                     dates.Modified,
				                     info.Length,
				                     info.Extension
					);
			}
			else {
				String fileName = result.Url.Segments[ result.Url.Segments.Length - 1 ];
				String fileExtension = ParseFileExtension( fileName );
				return new Document( fileName,
				                     result.Url.LocalPath,
				                     result.Snippet,
				                     dates.Created,
				                     dates.Modified,
				                     0,
				                     fileExtension
					);
			}
		}

		public IDocument Convert< T >( T result )
		{
			return Convert( result as GoogleSearchResult );
		}

		public IEnumerable< IDocument > Convert( IEnumerable< IGoogleSearchResult > results )
		{
			return Transform< IGoogleSearchResult, IDocument >(
				results,
				delegate( IGoogleSearchResult element ) { return Convert( element ) as Document; }
				);
		}

		//public IEnumerable<Tout> Convert<Tin, Tout>( IEnumerable<Tin> results ) where Tin : IGoogleSearchResult
		//{
		//    return (IEnumerable<Tout>)Convert( results as GoogleSearchResult );
		//}

		public IEnumerable< IDocument > ConvertFilesNamed( IEnumerable< IGoogleSearchResult > results, String onlyFilesNamed )
		{
			IEnumerable< IDocument > documents = Convert( results );
			_fileName = onlyFilesNamed;
			return Filter( documents, delegate( IDocument inputValue ) { return SameFileName( inputValue.FileName ); } );
		}

		public ICollection< IDocument > ConvertToCollection( ICollection< GoogleSearchResult > results )
		{
			ICollection< IDocument > documents = new Collection< IDocument >( );
			foreach( GoogleSearchResult result in results ) {
				documents.Add( Convert( result ) );
			}
			return documents;
		}

		public ICollection< IDocument > ConvertToCollection( ICollection< GoogleSearchResult > results, String fileName )
		{
			ICollection< IDocument > documents = new Collection< IDocument >( );
			foreach( GoogleSearchResult result in results ) {
				IDocument document = Convert( result );
				if( String.Compare( fileName, document.FileName, true ) == 0 ) {
					documents.Add( document );
				}
			}
			return documents;
		}

		#endregion

		#region Private Fields

		private delegate Tout Action< Tin, Tout >( Tin element );

		private delegate Boolean Predicate< T >( T inputValue );

		private String _fileName;

		private struct FileDates
		{
			#region Constructors

			public FileDates( DateTime created, DateTime modified )
			{
				_created = created;
				_modifed = modified;
			}

			#endregion

			#region Public Properties

			public DateTime Modified
			{
				get { return _modifed; }
			}

			public DateTime Created
			{
				get { return _created; }
			}

			#endregion

			#region Private Fields

			private DateTime _modifed;
			private DateTime _created;

			#endregion
		}

		#endregion

		#region Private Methods

		private static IEnumerable< Tout > Transform< Tin, Tout >( IEnumerable< Tin > list, Action< Tin, Tout > method )
		{
			foreach( Tin entry in list ) {
				yield return method( entry );
			}
		}

		private static IEnumerable< T > Filter< T >( IEnumerable< T > list, Predicate< T > condition )
		{
			foreach( T item in list ) {
				if( condition( item ) ) {
					yield return item;
				}
			}
		}

		private static String ParseFileExtension( String fileName )
		{
			return fileName.Substring( fileName.LastIndexOf( "." ) + 1 );
		}

		private static FileDates RetrieveDateUtc( String filePath )
		{
			try {
				if( !File.Exists( filePath ) ) {
					return new FileDates( DateTime.MinValue, DateTime.MinValue );
				}
				FileInfo info = new FileInfo( filePath );
				return new FileDates( info.CreationTimeUtc, info.LastWriteTimeUtc );
			}
			catch( NotSupportedException ) {
				return new FileDates( DateTime.MinValue, DateTime.MinValue );
			}
		}

		private Boolean SameFileName( String name )
		{
			if( !String.IsNullOrEmpty( _fileName ) ) {
				return _fileName == name;
			}
			return false;
		}

		#endregion
	}
}