/*
 * Created by: Mo Khan
 * Created: Friday, May 11, 2007
 */

using System;
using System.Data;
using System.Globalization;
using System.IO;
using DesktopSearch.Domain;

namespace DesktopSearch.Presentation
{
	public class DirectoryTreePresenter
	{
		#region Constructors

		public DirectoryTreePresenter( IDirectoryTreeView view, IAppSettings settings )
		{
			_view = view;
			_settings = settings;
			_view.InitializeTreeView += delegate { InitializeTreeTable( ); };
		}

		#endregion

		#region Private Fields

		private IDirectoryTreeView _view;
		private IAppSettings _settings;
		private Int32 _currentId;

		private class TreeColumns
		{
			private TreeColumns( ) {}
			public const String Id = "ID";
			public const String ParentId = "ParentID";
			public const String Text = "Text";
			public const String Url = "URL";
		}

		#endregion

		#region Private Methods

		private static DataTable BuildTreeViewTable( )
		{
			DataTable table = new DataTable( );
			table.Locale = CultureInfo.InvariantCulture;
			table.Columns.Add( TreeColumns.Id );
			table.Columns.Add( TreeColumns.ParentId );
			table.Columns.Add( TreeColumns.Text );
			table.Columns.Add( TreeColumns.Url );
			return table;
		}

		private void BindTreeTo( DataTable table )
		{
			_view.DirectoryTree.DataSource = table;
			_view.DirectoryTree.DataFieldID = TreeColumns.Id;
			_view.DirectoryTree.DataFieldParentID = TreeColumns.ParentId;
			_view.DirectoryTree.DataTextField = TreeColumns.Text;
			_view.DirectoryTree.DataValueField = TreeColumns.Url;
			_view.DirectoryTree.DataNavigateUrlField = TreeColumns.Url;
			_view.DirectoryTree.DataBind( );
		}

		private void FillRadTreeViewTable( String directoryPath, Int32? parentId, DataTable table )
		{
			DirectoryInfo dInfo = new DirectoryInfo( directoryPath );
			foreach( DirectoryInfo directory in dInfo.GetDirectories( ) ) {
				String url = directory.FullName.Replace( _settings.DocumentsPath, _settings.VirtualPath );
				table.Rows.Add(
					new String[]
						{
							_currentId.ToString( CultureInfo.InvariantCulture ), ( null == parentId ? null : parentId.ToString( ) ),
							directory.Name, url
						} );
				++_currentId;
				FillRadTreeViewTable( directory.FullName, _currentId, table );
			}
		}

		#endregion

		#region Event Handlers

		private void InitializeTreeTable( )
		{
			if( String.IsNullOrEmpty( _settings.DocumentsPath ) ) {
				throw new ArgumentNullException( );
			}
			DataTable table = BuildTreeViewTable( );
			_currentId = 1;
			FillRadTreeViewTable( _settings.DocumentsPath, null, table );

			BindTreeTo( table );
		}

		#endregion
	}
}